/*
 * Decompiled with CFR 0.152.
 */
package su.plo.voice.server.config;

import com.google.common.base.Charsets;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.Maps;
import java.io.BufferedReader;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.OpenOption;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentMap;
import java.util.stream.Collectors;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import su.plo.config.toml.Toml;
import su.plo.config.toml.TomlWriter;
import su.plo.voice.api.server.config.ResourceLoader;
import su.plo.voice.api.server.config.ServerLanguages;
import su.plo.voice.libs.crowdin.PlasmoCrowdinLib;

public final class VoiceServerLanguages
implements ServerLanguages {
    private static final Logger LOGGER = LoggerFactory.getLogger(VoiceServerLanguages.class);
    private final String defaultLanguageName;
    private final boolean crowdinDisabled;
    private final Map<String, VoiceServerLanguage> languages = Maps.newHashMap();

    @Override
    public synchronized void register(@NotNull ResourceLoader resourceLoader, @NotNull File languagesFolder) {
        try {
            HashMap languages = Maps.newHashMap();
            try (InputStream is = resourceLoader.load("languages/list");
                 BufferedReader br = new BufferedReader(new InputStreamReader(is, StandardCharsets.UTF_8));){
                for (String languageName : br.lines().filter(line -> !line.isEmpty()).collect(Collectors.toList())) {
                    File languageFile = new File(languagesFolder, String.format("%s.toml", languageName));
                    VoiceServerLanguage language = this.loadLanguage(resourceLoader, languageFile, languageName);
                    languages.put(languageName, language);
                }
            }
            this.register(languages, languagesFolder);
        }
        catch (IOException e) {
            throw new IllegalStateException("Failed to load languages", e);
        }
    }

    @Override
    public void register(@NotNull String crowdinProjectId, @Nullable String fileName, @NotNull ResourceLoader resourceLoader, @NotNull File languagesFolder) {
        CompletableFuture.runAsync(() -> this.registerSync(crowdinProjectId, fileName, resourceLoader, languagesFolder));
    }

    @Override
    public Map<String, String> getServerLanguage(@Nullable String languageName) {
        return this.getLanguage(languageName, "server");
    }

    @Override
    public Map<String, String> getClientLanguage(@Nullable String languageName) {
        return this.getLanguage(languageName, "client");
    }

    private void registerSync(@NotNull String crowdinProjectId, @Nullable String fileName, @NotNull ResourceLoader resourceLoader, @NotNull File languagesFolder) {
        try {
            File crowdinFolder;
            if (!this.crowdinDisabled) {
                try {
                    this.downloadCrowdinTranslations(crowdinProjectId, fileName, languagesFolder);
                }
                catch (Exception e) {
                    LOGGER.warn("Failed to download crowdin project {} ({}) translations: {}", new Object[]{crowdinProjectId, fileName, e.getMessage()});
                }
            }
            if (!(crowdinFolder = new File(languagesFolder, ".crowdin")).exists()) {
                this.register(resourceLoader, languagesFolder);
                return;
            }
            HashMap languages = Maps.newHashMap();
            File[] fileArray = null;
            try (InputStream is = resourceLoader.load("languages/list");
                 BufferedReader br = new BufferedReader(new InputStreamReader(is, StandardCharsets.UTF_8));){
                for (String languageName : br.lines().filter(line -> !line.isEmpty()).collect(Collectors.toList())) {
                    String languageFileName = String.format("%s.toml", languageName);
                    File crowdinFile = new File(crowdinFolder, languageFileName);
                    File languageFile = new File(languagesFolder, languageFileName);
                    VoiceServerLanguage language = this.loadLanguage(resourceLoader, crowdinFile, languageFile, languageName);
                    languages.put(languageName, language);
                }
            }
            catch (Throwable object) {
                fileArray = object;
                throw object;
            }
            File[] crowdinTranslations = crowdinFolder.listFiles();
            if (crowdinTranslations != null) {
                for (File crowdinFile : crowdinTranslations) {
                    String languageName;
                    if (crowdinFile.getName().equals("timestamp") || languages.containsKey(languageName = crowdinFile.getName().split("\\.")[0])) continue;
                    File languageFile = new File(languagesFolder, crowdinFile.getName());
                    VoiceServerLanguage language = this.loadLanguage(resourceLoader, crowdinFile, languageFile, languageName);
                    languages.put(languageName, language);
                }
            }
            this.register(languages, languagesFolder);
        }
        catch (IOException e) {
            throw new IllegalStateException("Failed to load languages", e);
        }
    }

    private void downloadCrowdinTranslations(@NotNull String crowdinProjectId, @Nullable String fileName, @NotNull File languagesFolder) throws Exception {
        long timestamp = 0L;
        File crowdinFolder = new File(languagesFolder, ".crowdin");
        File timestampFile = new File(crowdinFolder, "timestamp");
        if (timestampFile.exists()) {
            String timestampString = new String(Files.readAllBytes(timestampFile.toPath()));
            try {
                timestamp = Long.parseLong(timestampString);
            }
            catch (NumberFormatException numberFormatException) {
                // empty catch block
            }
        }
        if (System.currentTimeMillis() - timestamp < 259200000L) {
            return;
        }
        Map<String, byte[]> rawTranslations = PlasmoCrowdinLib.INSTANCE.downloadRawTranslations(crowdinProjectId, fileName).get();
        crowdinFolder.mkdirs();
        Files.write(timestampFile.toPath(), String.valueOf(System.currentTimeMillis()).getBytes(), new OpenOption[0]);
        if (rawTranslations.isEmpty()) {
            return;
        }
        for (Map.Entry<String, byte[]> entry : rawTranslations.entrySet()) {
            String languageName = entry.getKey();
            byte[] translationBytes = entry.getValue();
            Files.write(new File(crowdinFolder, String.format("%s.toml", languageName)).toPath(), translationBytes, new OpenOption[0]);
        }
    }

    private synchronized void register(@NotNull Map<String, VoiceServerLanguage> languages, @NotNull File languagesFolder) throws IOException {
        if (languages.size() == 0) {
            return;
        }
        VoiceServerLanguage defaultLanguage = languages.getOrDefault(this.defaultLanguageName, languages.get(languages.keySet().iterator().next()));
        languagesFolder.mkdirs();
        for (File file : languagesFolder.listFiles()) {
            String languageName2;
            String fileName;
            if (file.isDirectory() || !(fileName = file.getName()).endsWith(".toml") || languages.containsKey(languageName2 = fileName.substring(0, fileName.length() - 5))) continue;
            VoiceServerLanguage language2 = this.loadLanguage(file, null);
            if (defaultLanguage != null) {
                language2.merge(defaultLanguage);
            }
            languages.put(languageName2, language2);
        }
        for (Map.Entry entry : languages.entrySet()) {
            File languageFile = new File(languagesFolder, (String)entry.getKey() + ".toml");
            this.saveLanguage(languageFile, (VoiceServerLanguage)entry.getValue());
        }
        for (Map.Entry entry : languages.entrySet()) {
            VoiceServerLanguage language3 = this.languages.computeIfAbsent((String)entry.getKey(), key -> (VoiceServerLanguage)entry.getValue());
            language3.merge((VoiceServerLanguage)entry.getValue());
            language3.merge(defaultLanguage);
        }
        this.languages.forEach((languageName, language) -> {
            if (languages.containsKey(languageName)) {
                return;
            }
            language.merge(defaultLanguage);
        });
    }

    private Map<String, String> getLanguage(@Nullable String languageName, @NotNull String scope) {
        VoiceServerLanguage language = this.languages.get(languageName == null ? this.defaultLanguageName : languageName.toLowerCase());
        if (languageName == null && language == null) {
            return ImmutableMap.of();
        }
        if (language == null) {
            return this.getLanguage(null, scope);
        }
        return scope.equals("server") ? language.getServerLanguage() : language.getClientLanguage();
    }

    private VoiceServerLanguage loadLanguage(@NotNull ResourceLoader resourceLoader, @NotNull File crowdinTranslation, @NotNull File languageFile, @NotNull String languageName) throws IOException {
        Toml crowdinDefaults = null;
        if (crowdinTranslation.exists()) {
            try (FileInputStream fis = new FileInputStream(crowdinTranslation);){
                crowdinDefaults = new Toml().read(new InputStreamReader((InputStream)fis, Charsets.UTF_8));
            }
        }
        Toml jarDefaults = null;
        try (InputStream is2 = resourceLoader.load("languages/" + languageName + ".toml");){
            if (is2 != null) {
                jarDefaults = new Toml().read(new InputStreamReader(is2, Charsets.UTF_8));
            }
        }
        catch (Exception is2) {
            // empty catch block
        }
        if (crowdinDefaults == null && jarDefaults == null) {
            throw new IOException("Both crowdin and jar defaults are null for language " + languageName);
        }
        Toml tomlLanguage = new Toml();
        if (languageFile.exists()) {
            try {
                tomlLanguage.read(languageFile);
            }
            catch (Exception e) {
                throw new IOException("Failed to load language " + languageFile.getName(), e);
            }
        }
        VoiceServerLanguage language = new VoiceServerLanguage(tomlLanguage, null);
        if (crowdinDefaults != null) {
            language.merge(new VoiceServerLanguage(crowdinDefaults, null));
        }
        if (jarDefaults != null) {
            language.merge(new VoiceServerLanguage(jarDefaults, null));
        }
        return language;
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    private VoiceServerLanguage loadLanguage(@NotNull ResourceLoader resourceLoader, @NotNull File languageFile, @NotNull String languageName) throws IOException {
        try (InputStream is = resourceLoader.load("languages/" + languageName + ".toml");){
            Toml defaults = new Toml().read(new InputStreamReader(is, Charsets.UTF_8));
            VoiceServerLanguage voiceServerLanguage = this.loadLanguage(languageFile, defaults);
            return voiceServerLanguage;
        }
        catch (IOException e) {
            throw e;
        }
        catch (Exception e) {
            throw new IOException("Failed to load language " + languageName, e);
        }
    }

    private VoiceServerLanguage loadLanguage(@NotNull File languageFile, @Nullable Toml defaults) throws IOException {
        Toml language = new Toml();
        if (languageFile.exists()) {
            try {
                language.read(languageFile);
            }
            catch (Exception e) {
                throw new IOException("Failed to load language " + languageFile.getName(), e);
            }
        }
        return new VoiceServerLanguage(language, defaults);
    }

    private void saveLanguage(@NotNull File languageFile, @NotNull VoiceServerLanguage language) throws IOException {
        try {
            new TomlWriter().write(language.getOriginal(), languageFile);
        }
        catch (Exception e) {
            throw new IOException("Failed to save language", e);
        }
    }

    public VoiceServerLanguages(String defaultLanguageName, boolean crowdinDisabled) {
        this.defaultLanguageName = defaultLanguageName;
        this.crowdinDisabled = crowdinDisabled;
    }

    private static class VoiceServerLanguage {
        private Map<String, Object> original;
        private final Map<String, String> serverLanguage;
        private final Map<String, String> clientLanguage;

        public VoiceServerLanguage(@NotNull Toml language, @Nullable Toml defaults) {
            this.original = defaults == null ? language.toMap() : this.mergeMaps(language.toMap(), defaults.toMap());
            this.serverLanguage = this.mergeMaps(language, "server", defaults);
            this.clientLanguage = this.mergeMaps(language, "client", defaults);
        }

        public void merge(@NotNull VoiceServerLanguage language) {
            this.original = this.mergeMaps(this.original, language.original);
            language.serverLanguage.forEach(this.serverLanguage::putIfAbsent);
            language.clientLanguage.forEach(this.clientLanguage::putIfAbsent);
        }

        private Map<String, Object> mergeMaps(@NotNull Map<String, Object> language, @NotNull Map<String, Object> defaults) {
            ConcurrentMap merged = Maps.newConcurrentMap();
            merged.putAll(language);
            for (Map.Entry<String, Object> entry : defaults.entrySet()) {
                if (entry.getValue() instanceof Map) {
                    merged.put(entry.getKey(), this.mergeMaps((Map)language.getOrDefault(entry.getKey(), Maps.newHashMap()), (Map)entry.getValue()));
                    continue;
                }
                if (language.containsKey(entry.getKey())) continue;
                merged.put(entry.getKey(), entry.getValue());
            }
            return merged;
        }

        private Map<String, String> mergeMaps(@NotNull Toml language, @NotNull String scope, @Nullable Toml defaults) {
            ConcurrentMap defaultsMap = Maps.newConcurrentMap();
            if (defaults != null) {
                defaultsMap.putAll(this.languageToMapOfStrings(defaults.getTable(scope) == null ? new Toml() : defaults.getTable(scope)));
            }
            Map<String, String> languageMap = this.languageToMapOfStrings(language.getTable(scope) == null ? new Toml() : language.getTable(scope));
            defaultsMap.putAll(languageMap);
            return defaultsMap;
        }

        private Map<String, String> languageToMapOfStrings(@NotNull Toml language) {
            HashMap languageMap = Maps.newHashMap();
            language.toMap().forEach((key, value) -> {
                if (value instanceof Map) {
                    Map<String, String> tableContents = this.languageToMapOfStrings(language.getTable((String)key));
                    tableContents.forEach((contentKey, contentValue) -> languageMap.put(key + "." + contentKey, contentValue));
                } else {
                    languageMap.put(key, value.toString());
                }
            });
            return languageMap;
        }

        public Map<String, Object> getOriginal() {
            return this.original;
        }

        public Map<String, String> getServerLanguage() {
            return this.serverLanguage;
        }

        public Map<String, String> getClientLanguage() {
            return this.clientLanguage;
        }
    }
}

